"""
This model reparameterizes the core shell bicelle model. In this version the
SLD of the face and the rim are replaced with molecular constraints. Basically
the slds are replaced with the sum of SLDs from solvent and dry headgroups.
The SLDs are fixed to be the same in both shells, as expected, but the amount
of solvent in each or the rim and face shells are independent allowing the
amount of solvent in the curved rim to be higher (or lower) than that in the
flat face.

A further constraint would be to tie the SLD of the core to the amount of
amphiphile in the shells.  Indeed the total number of headgroups in all the
shells must be equal to the total number of tails in the core. To do this
would require replacing volume fraction with number of molecules. This in
turn would require a knowledge of molecular volumes of the head group and
tails (to replace the volume of the shells and core).

The new parameters are:
* volfrac_solv_face: The volume fraction of solvent in the face shells
* volfrac_solv_rim: The volume fraction of solvent in the rim shell
* sld_hg: The dry headgroup SLD

These replace the sld_face, the SLD of the face shells, and sld_rim, the SLD of
the rim shell.
"""
from numpy import inf
from sasmodels.core import reparameterize
parameters = [
    # name, units, default, [min, max], type, description
    ["volfrac_solv_face", "", 0.5, [0, 1], "", "water volume fraction in the face"],
    ["volfrac_solv_rim", "", 0.5, [0, 1], "", "water volume fraction in the rim"],
    ["sld_hg", "1e-6/Ang^2", 1, [-inf, inf], "sld", "sld of dry headgroup"],
]
translation = """
    sld_face = volfrac_solv_face * sld_solv + (1-volfrac_solv_face) * sld_hg
    sld_rim = volfrac_solv_rim * sld_solv + (1-volfrac_solv_rim) * sld_hg
    """
model_info = reparameterize('core_shell_bicelle', parameters, translation, __file__)